<?php
/**
 * Integrations Service.
 *
 * @package EverestForms\Pro
 * @since   1.3.7
 */

namespace EverestForms\Pro;

defined( 'ABSPATH' ) || exit;

use Kunnu\Dropbox\DropboxFile;
use Kunnu\Dropbox\Exceptions\DropboxClientException;

/**
 * Service class.
 */
class Service {

	/**
	 * The single instance of the class.
	 *
	 * @var object
	 */
	protected static $instance;

	/**
	 * Cloning is forbidden.
	 *
	 * @since 1.3.7
	 */
	public function __clone() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Cloning is forbidden.', 'everest-forms-pro' ), '1.3.7' );
	}

	/**
	 * Unserializing instances of this class is forbidden.
	 *
	 * @since 1.3.7
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Unserializing instances of this class is forbidden.', 'everest-forms-pro' ), '1.3.7' );
	}

	/**
	 * Main plugin class instance.
	 *
	 * Ensures only one instance of the plugin is loaded or can be loaded.
	 *
	 * @return object Main instance of the class.
	 */
	final public static function instance() {
		if ( is_null( static::$instance ) ) {
			static::$instance = new static();
		}
		return static::$instance;
	}

	/**
	 * Plugin Constructor.
	 */
	public function __construct() {
		add_filter( 'everest_forms_integrations', array( $this, 'add_integration' ) );

		// Integrations - Form settings.
		add_filter( 'everest_forms_integration_uploads', array( $this, 'dropbox_uploads' ), 5, 2 );
		add_filter( 'everest_forms_integration_uploads', array( $this, 'google_drive_uploads' ), 5, 2 );
		add_action( 'everest_forms_inline_integrations_settings', array( $this, 'integrations_settings' ), 10, 2 );
	}

	/**
	 * Register integration.
	 *
	 * @param array $integrations List of integrations.
	 */
	public function add_integration( $integrations ) {
		$integrations[] = 'EverestForms\Pro\Integrations\Dropbox';
		$integrations[] = 'EverestForms\Pro\Integrations\GoogleDrive';
		return $integrations;
	}

	/**
	 * Upload files to Dropbox.
	 *
	 * @param array $file      File data.
	 * @param array $form_data Form Data.
	 */
	public function dropbox_uploads( $file, $form_data ) {
		$client  = new \EverestForms\Pro\Integrations\Dropbox();
		$dropbox = $client->get_client();

		if ( isset( $form_data['settings']['dropbox_enabled'] ) && '1' === $form_data['settings']['dropbox_enabled'] ) {
			$path = isset( $form_data['settings']['dropbox_destination_path'] ) ? rtrim( $form_data['settings']['dropbox_destination_path'], '/' ) : '';

			// Check folder exists or create it.
			if ( ! empty( $path ) && '/' !== $path ) {
				try {
					// Get folder metadata.
					$destination_folder = $dropbox->getMetadata( $path );

					// If destination folder is not a folder, return current file URL.
					if ( 'folder' !== $destination_folder->{'.tag'} ) {
						$logger = evf_get_logger();
						$logger->error(
							esc_html__( 'Unable to upload file because destination is not a folder.', 'everest-forms-pro' ),
							array(
								'source' => 'file-upload',
							)
						);
					}
				} catch ( \Exception $e ) {
					// Folder does not exist. Try to create it.
					try {
						// Create folder.
						$dropbox->createFolder( $path );
					} catch ( \Exception $e ) {
						// Log that folder could not be created.
						$logger = evf_get_logger();
						$logger->error(
							sprintf( 'Unable to upload file because destination folder could not be created: %s', $e->getMessage() ),
							array(
								'source' => 'file-upload',
							)
						);
						return $file;
					}
				}
			}

			try {
				// Create Dropbox File from Path.
				$dropboxFile = new DropboxFile( $file['tmp_path'] );

				// Upload the file to Dropbox.
				$uploadedFile = $dropbox->upload( $dropboxFile, trailingslashit( $path ) . $file['file_name_new'], array( 'autorename' => true ) );

				try {
					$shareable_links = $dropbox->postToAPI(
						'/sharing/create_shared_link_with_settings',
						array(
							'path'     => $uploadedFile->getPathDisplay(),
							'settings' => array(
								'requested_visibility' => 'public',
							),
						)
					);

					$file_data = $shareable_links->getDecodedBody();

					// Get the dropbox URL to the file.
					if ( ! empty( $file_data['url'] ) ) {
						return array_merge(
							$file,
							array(
								'external' => 'dropbox',
								'file_url' => esc_url_raw( $file_data['url'] ),
							)
						);
					}
				} catch ( DropboxClientException $e ) {
					try {
						$shareable_links = $dropbox->postToAPI(
							'/sharing/list_shared_links',
							array(
								'path' => $uploadedFile->getPathDisplay(),
							)
						);

						// Get existing sharable links.
						$file_data = $shareable_links->getDecodedBody();

						// If links were found, return first link.
						if ( ! empty( $file_data['links'] ) ) {
							return array_merge(
								$file,
								array(
									'external' => 'dropbox',
									'file_url' => esc_url_raw( $file_data['links'][0]['url'] ),
								)
							);
						}
					} catch ( DropboxClientException $e ) {
						// Log that folder could not be created.
						$logger = evf_get_logger();
						$logger->error(
							/* translators: %s: message */
							sprintf( esc_html__( 'Unable to create shareable link for file: %s', 'everest-forms-pro' ), $e->getMessage() ),
							array(
								'source' => 'file-upload',
							)
						);
						return $file;
					}
				}
			} catch ( DropboxClientException $e ) {
				// Log that folder could not be created.
				$logger = evf_get_logger();
				$logger->error(
					$e->getMessage(),
					array(
						'source' => 'file-upload',
					)
				);
				return $file;
			}
		}

		return $file;
	}

	/**
	 * Upload files to Google Drive.
	 *
	 * @param array $file      File data.
	 * @param array $form_data Form Data.
	 */
	public function google_drive_uploads( $file, $form_data ) {
		$client  = new \EverestForms\Pro\Integrations\GoogleDrive();
		$service = new \Google_Service_Drive( $client->get_client() );

		if ( isset( $form_data['settings']['google_drive_enabled'] ) && '1' === $form_data['settings']['google_drive_enabled'] ) {
			try {
				$fileMetadata = $this->create_drive_file( $file['file_name_new'], $form_data['settings']['google_drive_destination_path'], $service );

				$content = file_get_contents( $file['tmp_path'] ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents

				$created_file = $service->files->create(
					$fileMetadata,
					array(
						'data'       => $content,
						'uploadType' => 'multipart',
						'fields'     => 'id',
					)
				);

				$response = $service->files->get( $created_file->id, array( 'fields' => 'id, webContentLink' ) );

				// Get the drive URL to the file.
				if ( $response && isset( $response->webContentLink ) ) {
					return array_merge(
						$file,
						array(
							'external' => 'google_drive',
							'file_url' => str_replace( array( '/uc?', '&export=download' ), array( '/open?', '' ), $response->webContentLink ),
						)
					);
				}
			} catch ( \Google_Exception $e ) {
				// Log that folder could not be created.
				$logger = evf_get_logger();
				$logger->error(
					$e->getMessage(),
					array(
						'source' => 'file-upload',
					)
				);
				return $file;
			}
		}

		return $file;
	}

	/**
	 * Create Google Drive file object.
	 *
	 * @param string $filename File name.
	 * @param string $path     File path.
	 * @param object $service  Drive service.
	 *
	 * @return \Google_Service_Drive_DriveFile
	 */
	public function create_drive_file( $filename, $path, $service ) {
		$args = array(
			'name' => $filename,
		);

		// Check folder exists or create it.
		if ( ! empty( $path ) && '/' !== $path ) {
			$folder_id       = $this->create_or_get_directory( $path, $service );
			$args['parents'] = array( $folder_id );
		}

		return new \Google_Service_Drive_DriveFile( $args );
	}

	/**
	 * Create or get directory.
	 *
	 * @param string $path    File path.
	 * @param object $service Drive service.
	 */
	public function create_or_get_directory( $path, $service ) {
		$directory_ids = $this->get_directory_id( $path, $service );

		$parent_id = null;
		foreach ( $directory_ids as $name => $file_id ) {
			if ( is_null( $file_id ) ) {
				// Create Dir.
				$args = array(
					'mimeType' => 'application/vnd.google-apps.folder',
					'name'     => $name,
				);

				if ( $parent_id ) {
					$args['parents'] = array( $parent_id );
				}
				$dir     = new \Google_Service_Drive_DriveFile( $args );
				$file    = $service->files->create( $dir );
				$file_id = $file->id;

				$directory_ids[ $name ] = $file_id;
			}

			$parent_id = $file_id;
		}

		return $parent_id;
	}

	/**
	 * Get directory IDs.
	 *
	 * @param string $path    File path.
	 * @param object $service Drive service.
	 */
	protected function get_directory_id( $path, $service ) {
		$directories   = array_filter( explode( '/', $path ) );
		$directory_ids = array_fill_keys( $directories, null );

		$current_directories = $this->get_all_directories( $service );

		$parents = array();
		foreach ( $directories as $key => $directory ) {
			$parent = isset( $parents[ $key ] ) ? $parents[ $key ] : null;

			$found_directory = $this->find_directory( $directory, $parent, $current_directories );
			if ( ! $found_directory ) {
				continue;
			}

			$directory_ids[ $directory ] = $found_directory;

			$next_key = $key + 1;
			if ( isset( $directories[ $next_key ] ) ) {
				$parents[ $key + 1 ] = $found_directory;
			}
		}

		return $directory_ids;
	}

	/**
	 * Get all Google Drive directories.
	 *
	 * @param object $service Drive service.
	 *
	 * @return array of Google drive directories.
	 */
	protected function get_all_directories( $service ) {
		$token           = '';
		$all_directories = array();

		do {
			$args = array(
				'fields'                    => 'nextPageToken, files(id, name, parents)',
				'q'                         => 'mimeType = \'application/vnd.google-apps.folder\' AND trashed != true',
				'pageSize'                  => 100,
				'supportsAllDrives'         => true,
				'includeItemsFromAllDrives' => true,
			);

			if ( $token ) {
				$args['pageToken'] = $token;
			}

			$directories     = $service->files->listFiles( $args );
			$all_directories = array_merge( $all_directories, $directories->getFiles() );
			$token           = $directories->getNextPageToken();
		} while ( ! is_null( $token ) );

		$directories = array();
		foreach ( $all_directories as $file ) {
			$id = $file->getID();

			$directories[ $id ] = array(
				'id'      => $id,
				'name'    => $file->getName(),
				'parents' => $file->getParents(),
			);
		}

		return $directories;
	}

	/**
	 * Find Google Drive directory.
	 *
	 * @param string $name        Directory name to find.
	 * @param string $parent      Parent ID of directory.
	 * @param array  $directories List of all directories.
	 *
	 * @return string|bool Found directory ID or false.
	 */
	protected function find_directory( $name, $parent, $directories ) {
		foreach ( $directories as $id => $directory ) {
			if ( strtolower( $name ) !== strtolower( $directory['name'] ) ) {
				continue;
			}

			if ( ! $parent && ( is_null( $directory['parents'] ) || ! isset( $directories[ $directory['parents'][0] ] ) ) ) {
				return $id;
			}

			if ( $parent && in_array( $parent, $directory['parents'], true ) ) {
				return $id;
			}
		}

		return false;
	}

	/**
	 * Integrations Settings.
	 *
	 * @param array $form_data Form Data.
	 * @param array $settings  Form Settings.
	 */
	public function integrations_settings( $form_data, $settings ) {
		echo '<div class="everest-forms-border-container"><h4 class="everest-forms-border-container-title">' . esc_html__( 'External Upload Path', 'everest-forms' ) . '</h4>';
		everest_forms_panel_field(
			'checkbox',
			'settings',
			'dropbox_enabled',
			$form_data,
			esc_html__( 'Dropbox File path', 'everest-forms' ),
			array(
				'default' => isset( $settings['dropbox_enabled'] ) ? $settings['dropbox_enabled'] : 0,
				'tooltip' => esc_html__( 'Custom directory for the files to be uploaded to in your Dropbox /Apps/Everest Forms/ folder.', 'everest-forms-pro' ),
			)
		);
		everest_forms_panel_field(
			'text',
			'settings',
			'dropbox_destination_path',
			$form_data,
			'',
			array(
				'class'   => isset( $settings['dropbox_enabled'] ) && '1' === $settings['dropbox_enabled'] ? '' : 'everest-forms-hidden',
				'default' => isset( $settings['dropbox_destination_path'] ) ? $settings['dropbox_destination_path'] : '',
			)
		);
		everest_forms_panel_field(
			'checkbox',
			'settings',
			'google_drive_enabled',
			$form_data,
			esc_html__( 'Google Drive File path', 'everest-forms' ),
			array(
				'default' => isset( $settings['google_drive_enabled'] ) ? $settings['google_drive_enabled'] : 0,
				'tooltip' => esc_html__( 'Custom directory for the files to be uploaded to in Google Drive.', 'everest-forms-pro' ),
			)
		);
		everest_forms_panel_field(
			'text',
			'settings',
			'google_drive_destination_path',
			$form_data,
			'',
			array(
				'class'   => isset( $settings['google_drive_enabled'] ) && '1' === $settings['google_drive_enabled'] ? '' : 'everest-forms-hidden',
				'default' => isset( $settings['google_drive_destination_path'] ) ? $settings['google_drive_destination_path'] : '',
			)
		);
		echo '</div>';
	}
}
