<?php
/**
 * Dropbox Integration.
 *
 * @package EverestForms\Pro\Integrations
 * @since   1.3.7
 */

namespace EverestForms\Pro\Integrations;

defined( 'ABSPATH' ) || exit;

use Kunnu\Dropbox\Dropbox as IntegrationDropbox;
use Kunnu\Dropbox\DropboxApp;

/**
 * Dropbox Integration.
 */
class Dropbox extends \EVF_Integration {

	/**
	 * Client.
	 *
	 * @var object
	 */
	public $client;

	/**
	 * Init and hook in the integration.
	 */
	public function __construct() {
		$this->id                 = 'dropbox';
		$this->icon               = plugins_url( '/assets/img/dropbox.png', EFP_PLUGIN_FILE );
		$this->method_title       = esc_html__( 'Dropbox', 'everest-forms-pro' );
		$this->method_description = esc_html__( 'Dropbox Integration with Everest Forms', 'everest-forms-pro' );

		// Load the settings.
		$this->init_settings();
		$this->init_form_fields();

		// OAuth 2.0 credentials.
		$this->client_id     = 'ZXF6MGllM2F2d3BrNXZq';
		$this->client_secret = 'ZDcydXJienJ3NXQ1eHdi';

		// Define user set variables.
		$this->auth_code      = $this->get_option( 'auth_code' );
		$this->access_token   = $this->get_option( 'access_token' );
		$this->integration    = $this->get_integration();
		$this->account_status = $this->is_auth_required() ? 'disconnected' : 'connected';

		// Get Google API client.
		if ( $this->is_integration_page() ) {
			$this->client = $this->get_client();
		}

		// Google Client API Authentication.
		add_action( 'everest_forms_integration_account_connect_' . $this->id, array( $this, 'api_authenticate' ) );
		add_action( 'everest_forms_integration_account_disconnect_' . $this->id, array( $this, 'api_deauthenticate' ) );
	}

	/**
	 * Initialize integration settings form fields.
	 */
	public function init_form_fields() {
		$this->form_fields = array(
			'auth_code' => array(
				'title'   => __( 'Enter Dropbox Access Code', 'everest-forms-pro' ),
				'type'    => 'text',
				'default' => '',
			),
		);
	}

	/**
	 * Is authentication required?
	 *
	 * @return bool
	 */
	public function is_auth_required() {
		return empty( $this->access_token );
	}

	/**
	 * Returns an authorized API client.
	 *
	 * @return \Dropbox the authorized client object
	 *
	 * @link https://github.com/kunalvarma05/dropbox-php-sdk/wiki/Authentication-and-Authorization
	 *
	 * @param bool $is_ajax If called from Ajax.
	 */
	public function get_client( $is_ajax = false ) {
		$logger = evf_get_logger();

		// Load client only once.
		if ( ! empty( $this->client ) && ! $is_ajax ) {
			return $this->client;
		}

		// Configure Dropbox application.
		$app = new DropboxApp( base64_decode( $this->client_id ), base64_decode( $this->client_secret ) ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode

		// For debugging only.
		if ( ! defined( 'EVF_DEBUG' ) ) {
			$guzzleClient = null;
		} else {
			$guzzleClient = new \GuzzleHttp\Client(
				array(
					'verify' => false, // Otherwise HTTPS requests will fail.
				)
			);
		}

		// Configure Dropbox service.
		$client = new IntegrationDropbox( $app, array( 'http_client_handler' => $guzzleClient ) );

		if ( $is_ajax ) {
			return $client;
		}

		if (
			! empty( $this->auth_code )
			&& $this->is_auth_required()
		) {
			try {
				$authHelper  = $client->getAuthHelper();
				$accessToken = $authHelper->getAccessToken( $this->auth_code );

				if ( ! empty( $accessToken->getToken() ) ) {
					$this->update_option( 'access_token', $accessToken->getToken() );
				}
			} catch ( \Exception $e ) {
				$accessToken['error'] = $e->getMessage();
				$logger->error(
					sprintf( 'Unable to fetch access token with auth code: %s', $accessToken['error'] ),
					array(
						'source' => 'dropbox',
					)
				);
			}
		}

		// Set the access token used for requests.
		if ( ! empty( $this->access_token ) ) {
			$client->setAccessToken( $this->access_token );
		}

		return $client;
	}

	/**
	 * Google sheets API authenticate.
	 *
	 * @param array $posted_data Posted client credentials.
	 */
	public function api_authenticate( $posted_data ) {
		$auth_code = $this->get_field_value( 'auth_code', $this->form_fields['auth_code'], $posted_data );

		// Is valid auth to proceed?
		if ( empty( $auth_code ) ) {
			wp_send_json_error(
				array(
					'error'     => esc_html__( 'Could not authenticate to the Google Sheets.', 'everest-forms-pro' ),
					'error_msg' => esc_html__( 'Please provide the correct Google access code.', 'everest-forms-pro' ),
				)
			);
		}

		try {
			$client      = $this->get_client( true );
			$authHelper  = $client->getAuthHelper();
			$accessToken = $authHelper->getAccessToken( $auth_code );

			if ( ! empty( $accessToken->getToken() ) ) {
				$this->update_option( 'auth_code', $auth_code );
				$this->update_option( 'access_token', $accessToken->getToken() );

				wp_send_json_success(
					array(
						'button'      => esc_html__( 'Remove Authentication', 'everest-forms-pro' ),
						'description' => esc_html__( 'Dropbox account authenticated.', 'everest-forms-pro' ),
					)
				);
			}
		} catch ( \Exception $e ) {
			wp_send_json_error(
				array(
					'error'     => esc_html__( 'Could not authenticate to the Dropbox.', 'everest-forms-pro' ),
					'error_msg' => ucfirst( json_decode( $e->getMessage() )->error_description ),
				)
			);
		}
	}

	/**
	 * Google sheets API authenticate.
	 *
	 * @param array $posted_data Posted client credentials.
	 */
	public function api_deauthenticate( $posted_data ) {
		$this->init_settings();

		$client = $this->get_client( true );

		if (
			empty( $posted_data['key'] )
			&& $this->id === $posted_data['source']
		) {
			update_option( $this->get_option_key(), array() );
			wp_send_json_success(
				array(
					'remove'      => false,
					'oauth'       => filter_var( $client->getAuthHelper()->getAuthUrl(), FILTER_SANITIZE_URL ),
					'button'      => esc_html__( 'Authenticate with Dropbox account', 'everest-forms-pro' ),
					'description' => esc_html__( 'Get the Dropbox access code, we will use that code to authenticate.', 'everest-forms-pro' ),
				)
			);
		}
	}

	/**
	 * Facilitates dropbox integration. Evoked from extensibility.
	 */
	public function output_integration() {
		?>
		<div class="everest-forms-integration-content">
			<div class="integration-addon-detail">
				<div class="evf-integration-info-header">
					<figure class="evf-integration-logo">
						<img src="<?php echo esc_attr( $this->icon ); ?>" alt="<?php echo esc_attr( 'Dropbox Icon' ); ?>">
					</figure>
					<div class="integration-info">
						<h3><?php echo $this->method_title; /* phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped */ ?></h3>
						<div class="integration-status <?php echo esc_attr( $this->account_status ); ?>">
							<span class="toggle-switch <?php echo esc_attr( $this->account_status ); ?>">
								<?php if ( 'connected' === $this->account_status ) : ?>
									<?php esc_html_e( 'Connected', 'everest-forms-pro' ); ?>
								<?php endif; ?>
							</span>
						</div>
					</div>
					</div>
				<p><?php echo $this->method_description; /* phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped */ ?></p>
			</div>
			<div class="integration-connection-detail">
				<div class="evf-account-connect">
					<h3><?php esc_attr_e( 'Authenticate Dropbox', 'everest-forms-pro' ); ?></h3>
					<?php if ( empty( $this->auth_code ) && $this->is_auth_required() ) : ?>
						<p><?php esc_html_e( 'Get the Dropbox access code, we will use that code to authenticate with Google.', 'everest-forms-pro' ); ?></p>
					<?php else : ?>
						<p><?php esc_html_e( 'Google sheets account authenticated.', 'everest-forms-pro' ); ?></p>
					<?php endif; ?>
					<form>
						<div class="evf-connection-form hidden">
							<input type="text" name="<?php echo esc_attr( $this->get_field_key( 'auth_code' ) ); ?>" id="<?php echo esc_attr( $this->get_field_key( 'auth_code' ) ); ?>" class="<?php echo esc_attr( $this->get_field_key( 'auth_code' ) ); ?>" placeholder="<?php esc_attr_e( 'Enter Dropbox Access Code', 'everest-forms-pro' ); ?>" value="<?php echo esc_attr( $this->auth_code ); ?>">
							<button type="submit" class="everest-forms-btn everest-forms-btn-primary everest-forms-integration-connect-account" style="padding:7px 14px" data-source="<?php echo esc_attr( $this->id ); ?>"><?php esc_html_e( 'Verify access code', 'everest-forms-pro' ); ?></button>
						</div>
						<?php if ( empty( $this->auth_code ) && $this->is_auth_required() ) : ?>
							<a href="<?php echo esc_url( filter_var( $this->client->getAuthHelper()->getAuthUrl(), FILTER_SANITIZE_URL ) ); ?>" class="everest-forms-btn everest-forms-btn-primary everest-forms-integration-open-window" data-source="<?php echo esc_attr( $this->id ); ?>">
								<?php esc_html_e( 'Authenticate with Dropbox account', 'everest-forms-pro' ); ?>
							</a>
						<?php else : ?>
							<a href="#" class="everest-forms-btn everest-forms-btn-secondary everest-forms-integration-disconnect-account" data-source="<?php echo esc_attr( $this->id ); ?>">
								<?php esc_html_e( 'Remove Authentication', 'everest-forms-pro' ); ?>
							</a>
						<?php endif; ?>
					</form>
				</div>
			</div>
		</div>
		<?php
	}
}
