<?php
/**
 * EverestForms Multi-Part setup
 *
 * @package EverestForms_MultiPart
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Main EverestForms Multi-Part Class.
 *
 * @class EverestForms_MultiPart
 */
final class EverestForms_MultiPart {

	/**
	 * Instance of this class.
	 *
	 * @var object
	 */
	protected static $instance = null;

	/**
	 * Initialize the plugin.
	 */
	private function __construct() {
		add_action( 'init', array( $this, 'load_plugin_textdomain' ) );
		add_filter( 'plugin_row_meta', array( $this, 'plugin_row_meta' ), 20, 2 );

		// Checks with Everest Forms is installed.
		if ( defined( 'EVF_VERSION' ) && version_compare( EVF_VERSION, '1.7.0', '>=' ) ) {

			// Checks with Everest Forms Pro is installed.
			if ( defined( 'EFP_VERSION' ) && version_compare( EFP_VERSION, '1.1', '>=' ) ) {
				$this->includes();

				// Hooks.
				add_action( 'init', array( $this, 'setup_parts' ), 0 );
				add_action( 'everest_forms_init', array( $this, 'plugin_updater' ) );
				add_action( 'wp_footer', array( $this, 'custom_inline_css' ), 5 );
				add_action( 'enqueue_block_assets', array( $this, 'enqueue_block_assets' ) );
				add_action( 'wp_enqueue_scripts', array( $this, 'frontend_enqueue_scripts' ) );
				add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );

				// Store parts.
				add_filter( 'everest_forms_parts_data', array( $this, 'multiparts_form_data' ), 10, 3 );
				add_filter( 'everest_forms_field_submit_visibility_class', array( $this, 'field_submit_visibility_class' ), 10, 2 );

				// Admin actions.
				add_action( 'everest_forms_builder_after_content_fields', array( $this, 'output_multipart_navigation' ) );
				add_action( 'everest_forms_builder_fields_tab', array( $this, 'add_multipart_fields_tab' ) );
				add_action( 'everest_forms_builder_fields_tab_content', array( $this, 'output_multipart_options' ) );

				// Builder actions.
				add_action( 'everest_forms_display_builder_fields_before', array( $this, 'display_builder_fields_before' ) );
				add_action( 'everest_forms_display_builder_fields_after', array( $this, 'display_builder_fields_after' ) );
				add_action( 'everest_forms_display_builder_row_after', array( $this, 'display_builder_row_after' ), 10, 3 );

				// Frontend actions.
				add_action( 'everest_forms_frontend_output', array( $this, 'output_part_indicator' ), 9 );
				add_action( 'everest_forms_display_fields_before', array( $this, 'display_fields_before' ) );
				add_action( 'everest_forms_display_fields_after', array( $this, 'display_fields_after' ) );
				add_action( 'everest_forms_display_row_after', array( $this, 'display_row_after' ), 10, 2 );
				add_action( 'everest_forms_display_submit_before', array( $this, 'display_part_buttons' ), 10, 2 );

				// AJAX events.
				add_action( 'wp_ajax_everest_forms_add_new_part', array( $this, 'add_new_part' ) );
			} else {
				add_action( 'admin_notices', array( $this, 'everest_forms_pro_missing_notice' ) );
			}
		} else {
			add_action( 'admin_notices', array( $this, 'everest_forms_missing_notice' ) );
		}
	}

	/**
	 * Return an instance of this class.
	 *
	 * @return object A single instance of this class.
	 */
	public static function get_instance() {
		// If the single instance hasn't been set, set it now.
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Load Localisation files.
	 *
	 * Note: the first-loaded translation file overrides any following ones if the same translation is present.
	 *
	 * Locales found in:
	 *      - WP_LANG_DIR/everest-forms-multi-part/everest-forms-multi-part-LOCALE.mo
	 *      - WP_LANG_DIR/plugins/everest-forms-multi-part-LOCALE.mo
	 */
	public function load_plugin_textdomain() {
		$locale = apply_filters( 'plugin_locale', get_locale(), 'everest-forms-multi-part' );

		load_textdomain( 'everest-forms-multi-part', WP_LANG_DIR . '/everest-forms-multi-part/everest-forms-multi-part-' . $locale . '.mo' );
		load_plugin_textdomain( 'everest-forms-multi-part', false, plugin_basename( dirname( EVF_MULTI_PART_PLUGIN_FILE ) ) . '/languages' );
	}

	/**
	 * Display row meta in the Plugins list table.
	 *
	 * @param  array  $plugin_meta Plugin Row Meta.
	 * @param  string $plugin_file Plugin Base file.
	 * @return array
	 */
	public function plugin_row_meta( $plugin_meta, $plugin_file ) {
		if ( plugin_basename( EVF_MULTI_PART_PLUGIN_FILE ) === $plugin_file ) {
			$new_plugin_meta = array(
				'docs' => '<a href="' . esc_url( 'https://docs.wpeverest.com/docs/everest-forms/everest-forms-add-ons/multi-part-form/' ) . '" aria-label="' . esc_attr__( 'View Everest Forms Multi-Part documentation', 'everest-forms-multi-part' ) . '">' . esc_html__( 'Docs', 'everest-forms-multi-part' ) . '</a>',
			);

			return array_merge( $plugin_meta, $new_plugin_meta );
		}

		return (array) $plugin_meta;
	}

	/**
	 * Includes.
	 */
	private function includes() {
		include_once dirname( __FILE__ ) . '/admin/class-evfmp-form-settings.php';
	}

	/**
	 * Plugin Updater.
	 */
	public function plugin_updater() {
		if ( class_exists( 'EVF_Plugin_Updater' ) ) {
			return EVF_Plugin_Updater::updates( EVF_MULTI_PART_PLUGIN_FILE, 5422, EVF_MULTI_PART_VERSION );
		}
	}

	/**
	 * Setup Multi-Part steps.
	 */
	public function setup_parts() {
		$form_id   = isset( $_GET['form_id'] ) ? absint( $_GET['form_id'] ) : 0; // phpcs:ignore WordPress.Security.NonceVerification
		$form_obj  = evf()->form->get( $form_id, array( 'content_only' ) );
		$form_data = ! empty( $form_obj->post_content ) ? evf_decode( $form_obj->post_content ) : '';
		$defaults  = array(
			0 => array(
				'id'   => 1,
				'name' => esc_html__( 'Part Title', 'everest-forms-multi-part' ),
			),
		);

		$this->parts = isset( $form_data['multi_part'] ) ? array_values( $form_data['multi_part'] ) : $defaults;
	}

	/**
	 * Enqueue scripts.
	 */
	public function admin_enqueue_scripts() {
		$screen    = get_current_screen();
		$screen_id = $screen ? $screen->id : '';
		$suffix    = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		// Register admin scripts.
		wp_register_style( 'everest-forms-multi-part-admin', plugins_url( '/assets/css/admin.css', EVF_MULTI_PART_PLUGIN_FILE ), array(), EVF_MULTI_PART_VERSION );
		wp_register_script( 'everest-forms-multi-part-admin', plugins_url( "/assets/js/admin/admin{$suffix}.js", EVF_MULTI_PART_PLUGIN_FILE ), array( 'jquery', 'jquery-confirm', 'jquery-ui-sortable', 'perfect-scrollbar' ), EVF_MULTI_PART_VERSION, true );

		// Add RTL support for admin styles.
		wp_style_add_data( 'everest-forms-multi-part-admin', 'rtl', 'replace' );

		// Admin styles for EVF pages only.
		if ( in_array( $screen_id, evf_get_screen_ids(), true ) ) {
			wp_enqueue_style( 'everest-forms-multi-part-admin' );
			wp_enqueue_script( 'everest-forms-multi-part-admin' );
			wp_localize_script(
				'everest-forms-multi-part-admin',
				'evf_multipart_admin',
				array(
					'ajax_url'                 => admin_url( 'admin-ajax.php' ),
					'add_new_part_nonce'       => wp_create_nonce( 'add-new-part' ),
					'i18n_part_title'          => esc_html__( 'Part Title', 'everest-forms-multi-part' ),
					'i18n_tooltip_msg'         => esc_html__( 'Drag and drop to set part order.', 'everest-forms-multi-part' ),
					'i18n_part_locked'         => esc_html__( 'Part Locked', 'everest-forms-multi-part' ),
					'i18n_part_locked_msg'     => esc_html__( 'Single part cannot be deleted.', 'everest-forms-multi-part' ),
					'i18n_delete_part_confirm' => esc_html__( 'Are you sure you want to delete this part?', 'everest-forms-multi-part' ),
				)
			);
		}
	}

	/**
	 * Add custom inline style.
	 */
	public function custom_inline_css() {
		$custom_css = array();

		if ( ! empty( EVF_Shortcode_Form::$parts ) ) {
			foreach ( EVF_Shortcode_Form::$parts as $form_id => $part_data ) {
				if ( ! empty( EVF_Shortcode_Form::$parts[ $form_id ]['settings'] ) ) {
					$settings               = $part_data['settings'];
					$custom_css[ $form_id ] = "
						#evf-form-{$form_id} .everest-forms-multi-part-indicator.progress .everest-forms-multi-part-indicator-progress {
							background-color: {$settings['indicator_color']} !important;
						}

						#evf-form-{$form_id} .everest-forms-multi-part-indicator.circles .everest-forms-multi-part--steps-list li.active {
							color: {$settings['indicator_color']} !important;
							border-color: {$settings['indicator_color']} !important;
						}

						#evf-form-{$form_id} .everest-forms-multi-part-indicator.arrow_steps .everest-forms-multi-part-indicator-title.active,
						#evf-form-{$form_id} .everest-forms-multi-part-indicator.circles .everest-forms-multi-part--steps-list li.active::before,
						#evf-form-{$form_id} .everest-forms-multi-part-indicator.circles .everest-forms-multi-part--steps-list li.active::after {
							background: {$settings['indicator_color']} !important;
						}

						#evf-form-{$form_id} .everest-forms-multi-part-indicator.arrow_steps .everest-forms-multi-part-indicator-title.active::before {
							border-left-color: {$settings['indicator_color']} !important;
						}

						#evf-form-{$form_id} .everest-forms-multi-part-indicator.arrow_steps .everest-forms-multi-part-indicator-title.active::after {
							border-top-color: {$settings['indicator_color']} !important;
							border-bottom-color: {$settings['indicator_color']} !important;
						}
					";
				}
			}
		}

		echo "<!-- EverestForms Multi-Part Colors -->\n";
		echo "<style type=\"text/css\">\n";
		echo implode( ' ', $custom_css ); // @codingStandardsIgnoreLine
		echo "\n</style>\n";
		echo "<!-- /EverestForms Multi-Part Colors -->\n";
	}

	/**
	 * Gutenberg block assets.
	 */
	public function enqueue_block_assets() {
		$this->custom_inline_css();
		wp_enqueue_style( 'everest-forms-multi-part', plugins_url( '/assets/css/everest-forms-multi-part.css', EVF_MULTI_PART_PLUGIN_FILE ), array(), EVF_MULTI_PART_VERSION );
	}

	/**
	 * Frontend Enqueue scripts.
	 */
	public function frontend_enqueue_scripts() {
		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		// Enqueue scripts.
		wp_enqueue_style( 'everest-forms-multi-part', plugins_url( '/assets/css/everest-forms-multi-part.css', EVF_MULTI_PART_PLUGIN_FILE ), array(), EVF_MULTI_PART_VERSION );
		wp_enqueue_script( 'everest-forms-multi-part', plugins_url( "/assets/js/frontend/everest-forms-multi-part{$suffix}.js", EVF_MULTI_PART_PLUGIN_FILE ), array( 'everest-forms' ), EVF_MULTI_PART_VERSION, true );
	}

	/**
	 * Returns information about parts if the form has multiple parts.
	 *
	 * @param  array $parts Form parts.
	 * @param  array $form_data Form data.
	 * @param  array $form_id Form ID.
	 * @return mixed false or an array of part data.
	 */
	public function multiparts_form_data( $parts, $form_data, $form_id ) {
		// Remove BW parts keys.
		$parts_remove_keys = array( 'total', 'current', 'parts', 'settings' );
		foreach ( $parts_remove_keys as $key ) {
			if ( ! empty( $parts ) && isset( $parts[ $key ] ) ) {
				unset( $parts[ $key ] );
			}
		}

		if ( isset( $form_data['settings']['enable_multi_part'] ) && evf_string_to_bool( $form_data['settings']['enable_multi_part'] ) ) {
			$settings = isset( $form_data['settings']['multi_part'] ) ? $form_data['settings']['multi_part'] : array();

			if ( ! empty( $form_data['multi_part'] ) ) {
				$parts[ $form_id ]['total']    = count( $form_data['multi_part'] );
				$parts[ $form_id ]['current']  = 1;
				$parts[ $form_id ]['parts']    = array_values( $form_data['multi_part'] );
				$parts[ $form_id ]['settings'] = wp_parse_args(
					$settings,
					array(
						'indicator'       => 'progress',
						'indicator_color' => '#7e3bd0',
						'nav_align'       => 'center',
					)
				);
			}
		}

		return isset( $parts[ $form_id ] ) ? $parts[ $form_id ] : array();
	}

	/**
	 * Add classes to submit field if the form has multiple parts.
	 *
	 * @param  array $classes Raw classes.
	 * @param  array $parts Form Multi-parts.
	 * @return array Array of field submit visibility classes.
	 */
	public function field_submit_visibility_class( $classes, $parts ) {
		if ( ! empty( $parts ) ) {
			$classes[] = 'everest-forms-multi-part-actions';

			if ( isset( $parts['settings']['nav_align'] ) ) {
				$classes[] = 'everest-forms-nav-align--' . sanitize_html_class( $parts['settings']['nav_align'] );
			}
		}

		return (array) $classes;
	}

	/**
	 * Output Multi-part navigation preview.
	 *
	 * @param object $form Form object.
	 */
	public function output_multipart_navigation( $form ) {
		?>
		<div class="everest-forms-multi-part-tabs evf-tabs-wrapper">
			<ul class="everest-forms-tabs">
			<?php foreach ( $this->parts as $part_index => $part ) : ?>
					<?php $part_index = absint( $part_index ) + 1; ?>
					<li class="part_<?php echo absint( $part_index ); ?>_data" data-part-id="<?php echo absint( $part_index ); ?>">
						<div class="tips sort" data-tip="<?php esc_attr_e( 'Drag and drop to set part order.', 'everest-forms-multi-part' ); ?>"></div>
						<a href="#part_<?php echo absint( $part_index ); ?>">
							<span><?php echo esc_html( $part['name'] ); ?></span>
						</a>
					</li>
				<?php endforeach; ?>
			</ul>
			<button type="button" class="button button-small add-new-part" data-total-parts="<?php echo count( $this->parts ); ?>"><?php esc_html_e( 'Add New Part', 'everest-forms-multi-part' ); ?></button>
		</div>
		<?php
	}

	/**
	 * Add Multi-Part fields tab.
	 */
	public function add_multipart_fields_tab() {
		?>
		<a href="#" id="multi-part-options" class="part-options" style="display: none"><?php esc_html_e( 'Page Options', 'everest-forms-multi-part' ); ?></a>
		<?php
	}

	/**
	 * AJAX add new part.
	 */
	public function add_new_part() {
		check_ajax_referer( 'add-new-part', 'security' );

		if ( ! current_user_can( 'manage_everest_forms' ) || ! isset( $_POST['total_parts'] ) ) {
			wp_die( -1 );
		}

		try {
			$total_parts   = absint( wp_unslash( $_POST['total_parts'] ) ); // WPCS: input var ok.
			$new_part_id   = $total_parts + 1;
			$new_part_name = esc_html__( 'Part Title', 'everest-forms-multi-part' );

			// Get multi-part panel fragment.
			ob_start();
			$this->multipart_elements_panel( $new_part_id );
			$multipart_elements_panel = ob_get_clean();

			// Get multi-part options fragment.
			ob_start();
			$this->multipart_elements(
				$new_part_id,
				array(
					'id'   => $new_part_id,
					'name' => $new_part_name,
				)
			);
			$multipart_elements_options = ob_get_clean();

			if ( $new_part_id > 1 ) {
				wp_send_json_success(
					array(
						'new_part_id' => $new_part_id,
						'fragments'   => apply_filters(
							'everest_forms_multi_part_fragments',
							array(
								'.evf-admin-field-wrapper' => $multipart_elements_panel,
								'.everest-forms-multi-part-options' => $multipart_elements_options,
							)
						),
					)
				);
			}
		} catch ( Exception $e ) {
			wp_send_json_error( array( 'error' => $e->getMessage() ) );
		}
	}

	/**
	 * Output Multi-part options.
	 */
	public function output_multipart_options() {
		?>
		<div class="everest-forms-multi-part-options">
		<?php
		if ( ! empty( $this->parts ) ) {
			foreach ( $this->parts as $part_index => $part ) {
				$part_index = absint( $part_index ) + 1;
				$this->multipart_elements( $part_index, $part );
			}
		}
		?>
		</div>
		<?php
	}

	/**
	 * Output Multi-part panel.
	 *
	 * @param int $part_id Part ID.
	 */
	private function multipart_elements_panel( $part_id ) {
		printf( '<div id="part_%1$s" class="panel everest-forms-part everest-forms-part-%1$s" data-part-id="%1$s"></div>', absint( $part_id ) );
	}

	/**
	 * Output Multi-part panel.
	 *
	 * @param int   $part_index Part index.
	 * @param array $part Array of part data.
	 */
	private function multipart_elements( $part_index, $part ) {
		$default_fields = array(
			'name' => array(
				'label'   => esc_html__( 'Part Title', 'everest-forms-multi-part' ),
				/* translators: %s: Part ID */
				'default' => sprintf( esc_html__( 'Part %s', 'everest-forms-multi-part' ), absint( $part['id'] ) ),
				'tooltip' => esc_html__( 'Enter text for the part title.', 'everest-forms-multi-part' ),
			),
			'next' => array(
				'label'   => esc_html__( 'Next Label', 'everest-forms-multi-part' ),
				'default' => esc_html__( 'Next', 'everest-forms-multi-part' ),
				'tooltip' => esc_html__( 'Enter text for Next part navigation button.', 'everest-forms-multi-part' ),
			),
			'prev' => array(
				'label'   => esc_html__( 'Previous Label', 'everest-forms-multi-part' ),
				'default' => esc_html__( 'Previous', 'everest-forms-multi-part' ),
				'tooltip' => esc_html__( 'Enter text for Previous part navigation button.', 'everest-forms-multi-part' ),
			),
		);
		?>
		<div class="everest-forms-multi-part-option" id="everest-forms-multi-part-option-<?php echo absint( $part_index ); ?>" data-part-id="<?php echo absint( $part_index ); ?>" >
			<input type="hidden" name="multi_part[part_<?php echo absint( $part_index ); ?>][id]" value="<?php echo absint( $part_index ); ?>" class="everest-forms-multi-part-option-hidden-id" />
			<div class="everest-forms-multi-part-option-group open" id="everest-forms-multi-part-option-<?php echo absint( $part_index ); ?>">
			<?php printf( '<a href="#" class="everest-forms-multi-part-option-toggle">%s<span> (ID #%s)</span><i class="handlediv"></i></a>', esc_html__( 'Multi-Part Options', 'everest-forms-multi-part' ), absint( $part_index ) ); ?>
				<div class="everest-forms-multi-part-option-group-inner">
				<?php foreach ( $default_fields as $field_key => $field ) : ?>
						<div class="everest-forms-multi-part-option-row everest-forms-multi-part-option-row-<?php echo esc_attr( $field_key ); ?>" id="everest-forms-multi-part-option-row-<?php echo esc_attr( $field_key ); ?>-label">
							<label for="everest-forms-multi-part-option-<?php echo esc_attr( $part_index . '-' . $field_key ); ?>"><?php echo esc_html( $field['label'] ); ?></label>
							<input type="text" class="widefat" id="everest-forms-multi-part-option-<?php echo esc_attr( $part_index . '-' . $field_key ); ?>" name="multi_part[part_<?php echo esc_attr( $part_index ); ?>][<?php echo esc_attr( $field_key ); ?>]" value="<?php echo isset( $part[ $field_key ] ) ? esc_attr( $part[ $field_key ] ) : esc_attr( $field['default'] ); ?>" />
						</div>
					<?php endforeach; ?>
				</div>
			</div>
			<div class="everest-forms-multi-part-delete">
				<a href="#" class="everest-forms-part-delete"><span class="dashicons dashicons-trash"></span> <?php esc_html_e( 'Delete Part', 'everest-forms-multi-part' ); ?></a>
			</div>
		</div>
		<?php
	}

	/**
	 * Output Multi-Part indicator.
	 *
	 * @param array $form_data Form data.
	 */
	public function output_part_indicator( $form_data ) {
		$form_id = $form_data['id'];

		if ( isset( $form_data['settings']['enable_multi_part'] ) && evf_string_to_bool( $form_data['settings']['enable_multi_part'] ) ) {
			$step     = 1;
			$parts    = EVF_Shortcode_Form::$parts[ $form_id ]['parts'];
			$settings = EVF_Shortcode_Form::$parts[ $form_id ]['settings'];

			// Output part steps indicator.
			if ( ! empty( $parts ) && 'none' !== $settings['indicator'] ) {
				printf( '<div class="everest-forms-multi-part-indicator %1$s" data-indicator="%1$s">', esc_attr( $settings['indicator'] ) );

				switch ( $settings['indicator'] ) {
					case 'circles':
						echo '<ol class="everest-forms-multi-part--steps-list">';

						// Circle theme.
						foreach ( $parts as $part ) {
							$class = ( 1 === $step ) ? 'active' : '';
							if ( ! empty( $part['name'] ) ) {
								printf( '<li class="everest-forms-multi-part-indicator-title %s">%s</li>', esc_attr( $class ), esc_html( evf_string_translation( $form_id, 'multi-part-title-' . $step, $part['name'] ) ) );
							}
							$step ++;
						}

						echo '</ol>';
						break;

					case 'arrow_steps':
						echo '<ul class="everest-forms-multi-part--steps-list">';

						// Arrow steps theme.
						foreach ( $parts as $part ) {
							$class = ( 1 === $step ) ? 'active' : '';
							if ( ! empty( $part['name'] ) ) {
								printf( '<li class="everest-forms-multi-part-indicator-title %s">%s</li>', esc_attr( $class ), esc_html( evf_string_translation( $form_id, 'multi-part-title-' . $step, $part['name'] ) ) );
							}
							$step ++;
						}

						echo '</ul>';
						break;

					case 'progress':
						$p1    = ! empty( $parts[0]['name'] ) ? esc_html( $parts[0]['name'] ) : '';
						$width = 100 / ( count( $parts ) ) . '%';
						$prog  = 'style="width:' . $width . ';"';
						$names = '';

						// Progress theme.
						foreach ( $parts as $part ) {
							if ( ! empty( $part['name'] ) ) {
								$names .= sprintf( 'data-part-%d-title="%s" ', $step, esc_attr( evf_string_translation( $form_id, 'multi-part-title-' . $step, $part['name'] ) ) );
							}
							$step ++;
						}

						echo '<div class="everest-forms-progressbar-status">';
						printf( '<span class="everest-forms-multi-part-indicator-title" %s>%s</span>', $names, evf_string_translation( $form_id, 'multi-part-title-1', $p1 ) ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
						printf(
							/* translators: %1$s - current step in multi-part form; %2$d - total number of pages. */
							'<span class="everest-forms-multi-part-indicator-steps">' . esc_html__( 'Step %1$s of %2$d', 'everest-forms-multi-part' ) . '</span>',
							'<span class="everest-forms-multi-part-indicator-steps-current">1</span>',
							count( $parts )
						);
						echo '</div>';
						printf( '<div class="everest-forms-multi-part-indicator-progress-wrap"><div class="everest-forms-multi-part-indicator-progress" %s></div></div>', $prog ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
						break;
				}

				do_action( 'everest_forms_part_indicator', $parts, $form_data );

				echo '</div>';
			}
		}
	}

	/**
	 * Display fields before.
	 */
	public function display_builder_fields_before() {
		echo '<div id="part_1" class="panel everest-forms-part everest-forms-part-1 first" data-part-id="1">';
	}

	/**
	 * Display fields after.
	 */
	public function display_builder_fields_after() {
		echo '</div>';
	}

	/**
	 * Display builder markup to end current part and begin the next.
	 *
	 * @param string $row_id    Row ID.
	 * @param array  $form_data Form Data.
	 * @param array  $form_id   Form ID.
	 */
	public function display_builder_row_after( $row_id, $form_data, $form_id ) {
		if ( isset( $form_id ) && ! empty( EVF_Builder_Fields::$parts[ $form_id ]['parts'] ) ) {
			$total   = EVF_Builder_Fields::$parts[ $form_id ]['total'];
			$current = EVF_Builder_Fields::$parts[ $form_id ]['current'];

			foreach ( EVF_Builder_Fields::$parts[ $form_id ]['parts'] as $part ) {
				if ( ! empty( $part['rows'] ) && end( $part['rows'] ) === $row_id ) {
					$next = $current + 1;
					$last = $next === $total ? 'last' : '';

					if ( $current !== $total ) {
						printf(
							'</div><div id="part_%1$s" class="panel everest-forms-part everest-forms-part-%1$s %2$s" data-part-id="%1$s" style="display:none;">',
							absint( $next ),
							esc_attr( $last )
						);
					}

					// Increase count for next part.
					EVF_Builder_Fields::$parts[ $form_id ]['current'] ++;
				}
			}
		}
	}

	/**
	 * Display fields before.
	 *
	 * @param array $form_data Form data.
	 */
	public function display_fields_before( $form_data ) {
		if ( isset( $form_data['settings']['enable_multi_part'] ) && evf_string_to_bool( $form_data['settings']['enable_multi_part'] ) ) {
			$parts = ! empty( EVF_Shortcode_Form::$parts ) ? true : false;

			if ( $parts ) {
				echo '<div class="everest-forms-part everest-forms-part-1 first">';
			}
		}
	}

	/**
	 * Display fields after.
	 *
	 * @param array $form_data Form data.
	 */
	public function display_fields_after( $form_data ) {
		if ( isset( $form_data['settings']['enable_multi_part'] ) && evf_string_to_bool( $form_data['settings']['enable_multi_part'] ) ) {
			$parts = ! empty( EVF_Shortcode_Form::$parts ) ? true : false;

			if ( $parts ) {
				echo '</div>';
			}
		}
	}

	/**
	 * Display frontend markup to end current part and begin the next.
	 *
	 * @param string $row_id    Row ID.
	 * @param array  $form_data Form Data.
	 */
	public function display_row_after( $row_id, $form_data ) {
		$form_id = $form_data['id'];

		if ( isset( $form_data['settings']['enable_multi_part'] ) && evf_string_to_bool( $form_data['settings']['enable_multi_part'] ) ) {
			$total   = EVF_Shortcode_Form::$parts[ $form_id ]['total'];
			$current = EVF_Shortcode_Form::$parts[ $form_id ]['current'];

			if ( ! empty( EVF_Shortcode_Form::$parts[ $form_id ]['parts'] ) ) {
				foreach ( EVF_Shortcode_Form::$parts[ $form_id ]['parts'] as $part ) {
					if ( ! empty( $part['rows'] ) && end( $part['rows'] ) === $row_id ) {
						$next = $current + 1;
						$last = $next === $total ? 'last' : '';

						if ( $current < $total ) {
								printf(
									'</div><div class="everest-forms-part everest-forms-part-%s %s" style="display:none;">',
									absint( $next ),
									esc_attr( $last )
								);
						}

						// Increase count for next part.
						EVF_Shortcode_Form::$parts[ $form_id ]['current'] ++;
					}
				}
			}
		}
	}

	/**
	 * Display part buttons.
	 *
	 * @param array $form_data Form Data.
	 */
	public function display_part_buttons( $form_data ) {
		$form_id              = $form_data['id'];
		$next_btn_labels_data = '';
		$prev_btn_labels_data = '';

		if ( ! empty( EVF_Shortcode_Form::$parts[ $form_id ]['parts'] ) ) {
			foreach ( EVF_Shortcode_Form::$parts[ $form_id ]['parts'] as $part_index => $part ) {
				// Register string for translation.
				if ( ! empty( $part['next'] ) ) {
					$part['next'] = evf_string_translation( $form_id, 'multi-part-next-label-' . absint( $part['id'] ), $part['next'] );
				}

				if ( ! empty( $part['prev'] ) ) {
					$part['prev'] = evf_string_translation( $form_id, 'multi-part-prev-label-' . absint( $part['id'] ), $part['prev'] );
				}

				if ( 0 === $part_index ) {
					$next_btn_label = ! empty( $part['next'] ) ? $part['next'] : '';
					$prev_btn_label = ! empty( $part['prev'] ) ? $part['prev'] : '';
				}

				$next_btn_labels_data .= sprintf( 'data-label-part-%d="%s" ', absint( $part['id'] ), esc_html( $part['next'] ) );
				$prev_btn_labels_data .= sprintf( 'data-label-part-%d="%s" ', absint( $part['id'] ), esc_html( $part['prev'] ) );
			}

			printf(
				'<button type="button" class="everest-forms-part-button everest-forms-part-prev button" data-action="prev" data-part="1" data-formid="%d" %s style="display:none;">%s</button>',
				absint( $form_data['id'] ),
				$prev_btn_labels_data, // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				esc_html( $prev_btn_label )
			);

			printf(
				'<button type="button" class="everest-forms-part-button everest-forms-part-next button" data-action="next" data-part="1" data-formid="%d" %s>%s</button>',
				absint( $form_data['id'] ),
				$next_btn_labels_data, // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				esc_html( $next_btn_label )
			);
		}
	}

	/**
	 * Everest Forms fallback notice.
	 */
	public function everest_forms_missing_notice() {
		/* translators: %s: everest-forms version */
		echo '<div class="error notice is-dismissible"><p>' . sprintf( esc_html__( 'Everest Forms - Multi-Part Forms requires at least %s or later to work!', 'everest-forms-multi-part' ), '<a href="https://wpeverest.com/wordpress-plugins/everest-forms/" target="_blank">' . esc_html__( 'Everest Forms 1.6.7', 'everest-forms-multi-part' ) . '</a>' ) . '</p></div>';
	}

	/**
	 * Everest Forms Pro fallback notice.
	 */
	public function everest_forms_pro_missing_notice() {
		/* translators: %s: everest-forms-pro version */
		echo '<div class="error notice is-dismissible"><p>' . sprintf( esc_html__( 'Everest Forms - Multi-Part Forms depends on the last version of %s or later to work!', 'everest-forms-multi-part' ), '<a href="https://wpeverest.com/wordpress-plugins/everest-forms/" target="_blank">' . esc_html__( 'Everest Forms Pro 1.1', 'everest-forms-multi-part' ) . '</a>' ) . '</p></div>';
	}
}
