<?php
/**
 * Captcha field
 *
 * @package EverestForms\Fields
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * EVF_Field_Captcha Class.
 */
class EVF_Field_Captcha extends EVF_Form_Fields {

	/**
	 * Math equation and operators.
	 *
	 * @var array
	 */
	public $math;

	/**
	 * Captcha questions to ask for.
	 *
	 * @var array
	 */
	public $questions;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->name     = esc_html__( 'Captcha', 'everest-forms-captcha' );
		$this->type     = 'captcha';
		$this->icon     = 'evf-icon evf-icon-captcha';
		$this->order    = 160;
		$this->group    = 'advanced';
		$this->settings = array(
			'basic-options'    => array(
				'field_options' => array(
					'label',
					'captcha',
					'description',
				),
			),
			'advanced-options' => array(
				'field_options' => array(
					'placeholder',
					'label_hide',
					'css',
				),
			),
		);

		// Allow customizing math captcha.
		$this->math = apply_filters(
			'everest_forms_math_captcha',
			array(
				'min' => 1,
				'max' => 15,
				'cal' => array( '+', '*' ),
			)
		);

		// Allow for additional questions or customizing captcha questions.
		$this->questions = apply_filters(
			'everest_forms_default_captcha_questions',
			array(
				1 => array(
					'question' => esc_html__( 'What is 2+3?', 'everest-forms-captcha' ),
					'answer'   => esc_html__( '5', 'everest-forms-captcha' ),
				),
				2 => array(
					'question' => '',
					'answer'   => '',
				),
			)
		);

		parent::__construct();
	}

	/**
	 * Hook in tabs.
	 */
	public function init_hooks() {
		add_action( 'wp_enqueue_scripts', array( $this, 'frontend_enqueue_scripts' ) );
		add_filter( 'everest_forms_process_after_filter', array( $this, 'process_remove_field' ) );
		add_filter( 'everest_forms_field_new_required', array( $this, 'field_default_required' ), 5, 3 );
		add_filter( 'everest_forms_field_properties_' . $this->type, array( $this, 'field_properties' ), 5, 3 );
	}

	/**
	 * Captcha Format and questions.
	 *
	 * @param array $field Field Data.
	 */
	public function captcha( $field ) {
		$format    = ! empty( $field['format'] ) ? esc_attr( $field['format'] ) : 'math';
		$questions = ! empty( $field['questions'] ) ? $field['questions'] : $this->questions;

		// Field is always required.
		$this->field_element(
			'text',
			$field,
			array(
				'type'  => 'hidden',
				'slug'  => 'required',
				'value' => '1',
			)
		);

		// Format.
		$format_label  = $this->field_element(
			'label',
			$field,
			array(
				'slug'    => 'format',
				'value'   => esc_html__( 'Format', 'everest-forms-captcha' ),
				'tooltip' => sprintf( esc_html__( 'Choose a captcha format to be displayed on frontend.', 'everest-forms-captcha' ) ),
			),
			false
		);
		$format_select = $this->field_element(
			'select',
			$field,
			array(
				'slug'    => 'format',
				'value'   => $format,
				'options' => array(
					'math'     => esc_html__( 'Math', 'everest-forms-captcha' ),
					'question' => esc_html__( 'Question and Answer', 'everest-forms-captcha' ),
				),
			),
			false
		);

		$this->field_element(
			'row',
			$field,
			array(
				'slug'    => 'format',
				'content' => $format_label . $format_select,
			)
		);

		// Questions.
		$questions_label = $this->field_element(
			'label',
			$field,
			array(
				'slug'    => 'questions',
				'value'   => esc_html__( 'Questions and Answers', 'everest-forms-captcha' ),
				'tooltip' => sprintf( esc_html__( 'Add multiple questions below to ask the user. We will select questions randomly.', 'everest-forms-captcha' ) ),
			),
			false
		);
		$questions_field = sprintf(
			'<ul data-next-id="%s" class="evf-questions-list" data-field-id="%s" data-field-type="%s">',
			max( array_keys( $questions ) ) + 1,
			esc_attr( $field['id'] ),
			esc_attr( $this->type )
		);
		foreach ( $questions as $key => $value ) {
			$questions_field .= '<li data-key="' . absint( $key ) . '">';
			$questions_field .= sprintf( '<div class="question-wrap"><input type="text" name="form_fields[%s][questions][%s][question]" value="%s" class="question" placeholder="%s"><a class="add" href="#"><i class="dashicons dashicons-plus"></i></a></div>', $field['id'], $key, esc_attr( $value['question'] ), esc_html__( 'Question', 'everest-forms-captcha' ) );
			$questions_field .= sprintf( '<div class="answer-wrap"><input type="text" name="form_fields[%s][questions][%s][answer]" value="%s" class="answer" placeholder="%s"><a class="remove" href="#"><i class="dashicons dashicons-minus"></i></a></div>', $field['id'], $key, esc_attr( $value['answer'] ), esc_html__( 'Answer', 'everest-forms-captcha' ) );
			$questions_field .= '</li>';
		}
		$questions_field .= '</ul>';

		$this->field_element(
			'row',
			$field,
			array(
				'slug'    => 'questions',
				'content' => $questions_label . $questions_field,
				'class'   => 'math' === $format ? 'everest-forms-hidden' : '',
			)
		);
	}

	/**
	 * Frontend Enqueue scripts.
	 */
	public function frontend_enqueue_scripts() {
		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		// Enqueue scripts.
		wp_enqueue_script( 'everest-forms-custom-captcha', plugins_url( "/assets/js/frontend/everest-forms-custom-captcha{$suffix}.js", EVF_CAPTCHA_PLUGIN_FILE ), array( 'jquery', 'everest-forms' ), EverestForms_Captcha::VERSION, true );
		wp_localize_script(
			'everest-forms-custom-captcha',
			'everest_forms_captcha_params',
			array(
				'max'                 => $this->math['max'],
				'min'                 => $this->math['min'],
				'cal'                 => $this->math['cal'],
				'i18n_messages_error' => esc_html__( 'Incorrect answer.', 'everest-forms-captcha' ),
			)
		);
	}

	/**
	 * Field should default to being required.
	 *
	 * @since 1.0.0
	 *
	 * @param bool  $required Required status, true is required.
	 * @param array $field    Field settings.
	 *
	 * @return bool
	 */
	public function field_default_required( $required, $field ) {
		if ( 'captcha' === $field['type'] ) {
			return true;
		}

		return $required;
	}

	/**
	 * Don't store captcha fields since it's for validation only.
	 *
	 * @since 1.0.0
	 *
	 * @param  array $fields Field settings.
	 * @return array
	 */
	public function process_remove_field( $fields ) {
		foreach ( $fields as $id => $field ) {
			// Remove captcha from saved data.
			if ( 'captcha' === $field['type'] ) {
				unset( $fields[ $id ] );
			}
		}

		return $fields;
	}

	/**
	 * Define additional field properties.
	 *
	 * @since 1.0.0
	 *
	 * @param array $properties Field properties.
	 * @param array $field      Field settings.
	 * @param array $form_data  Form data and settings.
	 *
	 * @return array
	 */
	public function field_properties( $properties, $field, $form_data ) {
		$field_id = $field['id'];
		$format   = ! empty( $field['format'] ) ? $field['format'] : 'math';

		// Input Primary: adjust name.
		$properties['inputs']['primary']['attr']['name'] = "everest_forms[form_fields][{$field_id}][a]";

		// Input Primary: adjust class.
		$properties['inputs']['primary']['class'][] = 'a';

		// Input Primary: adjust value.
		$properties['inputs']['primary']['attr']['value'] = '';

		// Input Primary: type data attr.
		$properties['inputs']['primary']['data']['rule-evf-captcha'] = $format;

		return $properties;
	}

	/**
	 * Field preview inside the builder.
	 *
	 * @since 1.0.0
	 *
	 * @param array $field Field settings.
	 */
	public function field_preview( $field ) {
		$placeholder = ! empty( $field['placeholder'] ) ? esc_attr( $field['placeholder'] ) : '';
		$format      = ! empty( $field['format'] ) ? $field['format'] : 'math';
		$number1     = wp_rand( $this->math['min'], $this->math['max'] );
		$number2     = wp_rand( $this->math['min'], $this->math['max'] );
		$cal         = $this->math['cal'][ wp_rand( 0, count( $this->math['cal'] ) - 1 ) ];
		$questions   = ! empty( $field['questions'] ) ? $field['questions'] : $this->questions;
		$question    = current( $questions );

		// Label.
		$this->field_preview_option( 'label', $field );
		?>
		<div class="format-selected format-selected-<?php echo esc_attr( $format ); ?>">
			<span class="everest-forms-equation">
				<?php printf( '%s %s %s = ', $number1, $cal, $number2 ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
			</span>
			<p class="everest-forms-question"><?php echo esc_html( $question['question'] ); ?></p>
			<input type="text" placeholder="<?php echo esc_attr( $placeholder ); ?>" class="widefat" disabled>
		</div>
		<?php
		// Description.
		$this->field_preview_option( 'description', $field );
	}

	/**
	 * Field display on the form front-end.
	 *
	 * @since 1.0.0
	 *
	 * @param array $field      Field settings.
	 * @param array $field_atts Field attributes.
	 * @param array $form_data  Form data and settings.
	 */
	public function field_display( $field, $field_atts, $form_data ) {
		$field_id = $field['id'];
		$primary  = $field['properties']['inputs']['primary'];
		$format   = $form_data['form_fields'][ $field_id ]['format'];

		if ( 'math' === $format ) {
			// Math Captcha.
			?>
			<div class="everest-forms-captcha-math">
				<span class="everest-forms-captcha-equation">
					<span class="n1"></span>
					<span class="cal"></span>
					<span class="n2"></span>
					<span class="e">=</span>
				</span>
				<?php
				printf(
					'<input type="text" %s %s>',
					evf_html_attributes( $primary['id'], $primary['class'], $primary['data'], $primary['attr'] ),
					$primary['required'] // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				);
				?>
				<input type="hidden" name="everest_forms[form_fields][<?php echo esc_attr( $field_id ); ?>][cal]" class="cal">
				<input type="hidden" name="everest_forms[form_fields][<?php echo esc_attr( $field_id ); ?>][n2]" class="n2">
				<input type="hidden" name="everest_forms[form_fields][<?php echo esc_attr( $field_id ); ?>][n1]" class="n1">
			</div>
			<?php
		} else {
			// Question and Answer captcha.
			$qid = $this->random_question( $field, $form_data );
			$q   = esc_html( $form_data['form_fields'][ $field_id ]['questions'][ $qid ]['question'] );
			$a   = esc_attr( $form_data['form_fields'][ $field_id ]['questions'][ $qid ]['answer'] );
			?>
			<p class="everest-forms-captcha-question"><?php echo esc_html( $q ); ?></p>
			<?php
			$primary['data']['a'] = $a;
			printf(
				'<input type="text" %s %s>',
				evf_html_attributes( $primary['id'], $primary['class'], $primary['data'], $primary['attr'] ),
				$primary['required'] // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			);
			?>
			<input type="hidden" name="everest_forms[form_fields][<?php echo esc_attr( $field_id ); ?>][q]" value="<?php echo esc_attr( $qid ); ?>">
			<?php
		}
	}

	/**
	 * Select a random question.
	 *
	 * @since 1.0.0
	 *
	 * @param array $field     Field settings.
	 * @param array $form_data Form data and settings.
	 *
	 * @return bool|int
	 */
	public function random_question( $field, $form_data ) {
		$field_id    = $field['id'];
		$form_fields = $form_data['form_fields'][ $field_id ];

		if ( empty( $form_fields['questions'] ) ) {
			return false;
		}

		$index = array_rand( $form_fields['questions'] );

		if ( ! isset( $form_fields['questions'][ $index ]['question'] ) || ! isset( $field['questions'][ $index ]['answer'] ) ) {
			$index = $this->random_question( $form_fields, $form_data );
		}

		return $index;
	}

	/**
	 * Validates field on form submit.
	 *
	 * @since 1.0.0
	 *
	 * @param int   $field_id     Field ID.
	 * @param array $field_submit Submitted field value.
	 * @param array $form_data    Form data and settings.
	 */
	public function validate( $field_id, $field_submit, $form_data ) {
		$form_id = absint( $form_data['id'] );
		$format  = $form_data['form_fields'][ $field_id ]['format'];

		/**
		 * Check for conditional logic status.
		 *
		 * @todo Make this fix by resetting form elements that are conditionally hidden via JS.
		 */
		if ( isset( $form_data['form_fields'][ $field_id ]['conditional_logic_status'] ) && '1' === $form_data['form_fields'][ $field_id ]['conditional_logic_status'] ) {
			return;
		}

		// Math captcha.
		if ( 'math' === $format ) {
			// All calculation fields are required.
			if ( ( ( empty( $field_submit['a'] ) && '0' !== $field_submit['a'] ) || empty( $field_submit['n1'] ) || empty( $field_submit['cal'] ) || empty( $field_submit['n2'] ) ) ) {
				evf()->task->errors[ $form_id ][ $field_id ] = evf_get_required_label();
				return;
			}

			$n1  = $field_submit['n1'];
			$cal = $field_submit['cal'];
			$n2  = $field_submit['n2'];
			$a   = (int) trim( $field_submit['a'] );
			$x   = false;

			switch ( $cal ) {
				case '+':
					$x = ( $n1 + $n2 );
					break;
				case '-':
					$x = ( $n1 - $n2 );
					break;
				case '*':
					$x = ( $n1 * $n2 );
					break;
			}

			if ( $x !== $a ) {
				evf()->task->errors[ $form_id ][ $field_id ] = esc_html__( 'Incorrect answer', 'everest-forms-captcha' );
				return;
			}
		}

		// Question answer captcha.
		if ( 'question' === $format ) {
			// All fields are required.
			if ( ( empty( $field_submit['q'] ) || ( empty( $field_submit['a'] ) && '0' !== $field_submit['a'] ) ) ) {
				evf()->task->errors[ $form_id ][ $field_id ] = evf_get_required_label();
				return;
			}

			if ( strtolower( trim( $field_submit['a'] ) ) !== strtolower( trim( $form_data['form_fields'][ $field_id ]['questions'][ $field_submit['q'] ]['answer'] ) ) ) {
				evf()->task->errors[ $form_id ][ $field_id ] = esc_html__( 'Incorrect answer', 'everest-forms-captcha' );
				return;
			}
		}
	}
}
