<?php
/**
 * EverestForms Captcha setup
 *
 * @package EverestForms_Captcha
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Main EverestForms Captcha Class.
 *
 * @class EverestForms_Captcha
 */
final class EverestForms_Captcha {

	/**
	 * Plugin version.
	 *
	 * @var string
	 */
	const VERSION = '1.0.1';

	/**
	 * Instance of this class.
	 *
	 * @var object
	 */
	protected static $instance = null;

	/**
	 * Initialize the plugin.
	 */
	private function __construct() {
		// Load plugin text domain.
		add_action( 'init', array( $this, 'load_plugin_textdomain' ) );

		// Checks with Everest Forms Pro is installed.
		if ( defined( 'EFP_VERSION' ) && version_compare( EFP_VERSION, '1.2.0', '>=' ) ) {
			$this->includes();

			// Hooks.
			add_action( 'everest_forms_init', array( $this, 'plugin_updater' ) );
			add_filter( 'everest_forms_fields', array( $this, 'form_fields' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
			add_filter( 'plugin_row_meta', array( $this, 'plugin_row_meta' ), 20, 2 );
		} else {
			add_action( 'admin_notices', array( $this, 'everest_forms_pro_missing_notice' ) );
		}
	}

	/**
	 * Return an instance of this class.
	 *
	 * @return object A single instance of this class.
	 */
	public static function get_instance() {
		// If the single instance hasn't been set, set it now.
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Load Localisation files.
	 *
	 * Note: the first-loaded translation file overrides any following ones if the same translation is present.
	 *
	 * Locales found in:
	 *      - WP_LANG_DIR/everest-forms-captcha/everest-forms-captcha-LOCALE.mo
	 *      - WP_LANG_DIR/plugins/everest-forms-captcha-LOCALE.mo
	 */
	public function load_plugin_textdomain() {
		$locale = apply_filters( 'plugin_locale', get_locale(), 'everest-forms-captcha' );

		load_textdomain( 'everest-forms-captcha', WP_LANG_DIR . '/everest-forms-captcha/everest-forms-captcha-' . $locale . '.mo' );
		load_plugin_textdomain( 'everest-forms-captcha', false, plugin_basename( dirname( EVF_CAPTCHA_PLUGIN_FILE ) ) . '/languages' );
	}

	/**
	 * Includes.
	 */
	private function includes() {
		include_once dirname( __FILE__ ) . '/class-evf-field-captcha.php';
	}

	/**
	 * Plugin Updater.
	 */
	public function plugin_updater() {
		if ( class_exists( 'EVF_Plugin_Updater' ) ) {
			return EVF_Plugin_Updater::updates( EVF_CAPTCHA_PLUGIN_FILE, 22441, self::VERSION );
		}
	}

	/**
	 * Load captcha fields available in the addon.
	 *
	 * @param  array $fields Registered form fields.
	 * @return array
	 */
	public function form_fields( $fields ) {
		if ( defined( 'EVF_VERSION' ) && version_compare( EVF_VERSION, '1.6.4', '<=' ) ) {
			return array_merge( $fields, array( 'EVF_Field_Captcha' ) );
		}

		return $fields;
	}

	/**
	 * Enqueue scripts.
	 */
	public function admin_enqueue_scripts() {
		$screen    = get_current_screen();
		$screen_id = $screen ? $screen->id : '';
		$suffix    = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		// Register admin scripts.
		wp_register_script( 'everest-forms-custom-captcha-admin', plugins_url( "/assets/js/admin/admin{$suffix}.js", EVF_CAPTCHA_PLUGIN_FILE ), array( 'jquery' ), self::VERSION, true );

		// Admin scripts for EVF builder page.
		if ( 'everest-forms_page_evf-builder' === $screen_id ) {
			wp_enqueue_script( 'everest-forms-custom-captcha-admin' );
		}
	}

	/**
	 * Display row meta in the Plugins list table.
	 *
	 * @param  array  $plugin_meta Plugin Row Meta.
	 * @param  string $plugin_file Plugin Base file.
	 * @return array
	 */
	public function plugin_row_meta( $plugin_meta, $plugin_file ) {
		if ( plugin_basename( EVF_CAPTCHA_PLUGIN_FILE ) === $plugin_file ) {
			$new_plugin_meta = array(
				'docs' => '<a href="' . esc_url( 'https://docs.wpeverest.com/docs/everest-forms/everest-forms-add-ons/captcha/' ) . '" aria-label="' . esc_attr__( 'View Everest Forms Custom Captcha documentation', 'everest-forms-captcha' ) . '">' . esc_html__( 'Docs', 'everest-forms-captcha' ) . '</a>',
			);

			return array_merge( $plugin_meta, $new_plugin_meta );
		}

		return (array) $plugin_meta;
	}

	/**
	 * Everest Forms Pro fallback notice.
	 */
	public function everest_forms_pro_missing_notice() {
		/* translators: %s: everest-forms-pro version */
		echo '<div class="error notice is-dismissible"><p>' . sprintf( esc_html__( 'Everest Forms - Custom Captcha depends on the last version of %s or later to work!', 'everest-forms-captcha' ), '<a href="https://wpeverest.com/wordpress-plugins/everest-forms/" target="_blank">' . esc_html__( 'Everest Forms Pro 1.2.0', 'everest-forms-captcha' ) . '</a>' ) . '</p></div>';
	}
}
