<?php
/**
 * Save Form Dataclass.
 *
 * @package EverestForms\SaveAndContinue
 * @since   1.0.0
 */

namespace EverestForms\SaveAndContinue;

/**
 * Save and Continue Feature Class.
 *
 * @since 1.0.0
 */
class SaveEntry extends \EVF_Form_Task {

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'wp_ajax_nopriv_everest_forms_sc_form_submission', array( $this, 'save_entry' ) );
		add_action( 'wp_ajax_everest_forms_sc_form_submission', array( $this, 'save_entry' ) );
		add_filter( 'everest_forms_entry_data', array( $this, 'entry_data' ), 5, 2 );
	}

	/**
	 * Entry Data.
	 *
	 * @param mixed $data Entry Data.
	 * @param mixed $entry Entry.
	 *
	 * @since 1.0.0
	 */
	public function entry_data( $data, $entry ) {
		if ( ! empty( $entry['token'] ) ) {
			$data['status'] = 'draft';
			$data['token']  = $entry['token'];
		}
		return $data;
	}

	/**
	 * Save form entry.
	 *
	 * @since 1.0.0
	 */
	public function save_entry() {
		// Action Check.
		if ( empty( $_POST['everest_forms']['id'] ) || empty( $_POST['action'] ) || 'everest_forms_sc_form_submission' !== $_POST['action'] ) { // phpcs:ignore WordPress.Security.NonceVerification
			return;
		}
		try {
			$entry = stripslashes_deep( $_POST['everest_forms'] ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			$this->errors           = array();
			$this->form_fields      = array();
			$form_id                = absint( $entry['id'] );
			$form                   = evf()->form->get( $form_id );
			$honeypot               = false;
			$response_data          = array();
			$this->evf_notice_print = false;

			$entry['url']   = esc_url( $entry['url'] );
			$entry['token'] = substr( md5( time() . wp_rand( 0, 10 ) ), 0, 32 );

			// Check nonce for form submission.
			if ( empty( $_POST['_wpnonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['_wpnonce'] ), 'everest-forms_process_submit' ) ) { // phpcs:ignore WordPress.Security.NonceVerification, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				$this->errors[] = esc_html__( 'We were unable to process your form, please try again.', 'everest-forms-save-and-continue' );
			}

			// Validate form is real and active (published).
			if ( ! $form || 'publish' !== $form->post_status ) {
				$this->errors[] = esc_html__( 'Invalid form. Please check again.', 'everest-forms-save-and-continue' );
			}

			// Formatted form data for hooks.
			$this->form_data = apply_filters( 'everest_forms_process_before_form_data', evf_decode( $form->post_content ), $entry );

			// Pre-process/validate hooks and filter. Data is not validated or cleaned yet so use with caution.
			$entry                      = apply_filters( 'everest_forms_process_before_filter', $entry, $this->form_data );
			$this->form_data['page_id'] = array_key_exists( 'post_id', $entry ) ? $entry['post_id'] : $form_id;

			// Prepare fields for entry_save.
			foreach ( $this->form_data['form_fields'] as $field ) {
				if ( '' === isset( $this->form_data['form_fields']['meta-key'] ) ) {
					continue;
				}

				$field_id     = $field['id'];
				$field_type   = $field['type'];
				$field_submit = isset( $entry['form_fields'][ $field_id ] ) ? $entry['form_fields'][ $field_id ] : '';

				if ( 'signature' === $field_type ) {
					$field_submit = isset( $field_submit['signature_image'] ) ? $field_submit['signature_image'] : '';
				}

				$exclude = array( 'title', 'html', 'captcha', 'image-upload', 'file-upload' );

				if ( ! in_array( $field_type, $exclude, true ) ) {
					$this->form_fields[ $field_id ] = array(
						'id'       => $field_id,
						'name'     => sanitize_text_field( $field['label'] ),
						'meta_key' => $this->form_data['form_fields'][ $field_id ]['meta-key'],
						'type'     => $field_type,
						'value'    => evf_sanitize_textarea_field( $field_submit ),
					);
				}
			}

			// If validation issues occur, send the results accordingly.
			if ( count( $this->errors ) ) {
				// Error response.
				wp_send_json_error(
					array(
						'title'   => __( 'Form has not been submitted, please see the errors below.', 'everest-forms-save-and-continue' ),
						'message' => implode( '<br>', $this->errors ),
					)
				);
			}

			// Pass the form created date into the form data.
			$this->form_data['created'] = $form->post_date;

			// Setting data to evf task.
			evf()->task->form_fields = $this->form_fields;
			// Format fields.
			foreach ( (array) $this->form_data['form_fields'] as $field ) {
				$field_id     = $field['id'];
				$field_key    = isset( $field['meta-key'] ) ? $field['meta-key'] : '';
				$field_type   = $field['type'];
				$field_submit = isset( $entry['form_fields'][ $field_id ] ) ? $entry['form_fields'][ $field_id ] : '';
				do_action( "everest_forms_process_format_{$field_type}", $field_id, $field_submit, $this->form_data, $field_key );
			}

			// Getting formatted data from evf task.
			$this->form_fields = evf()->task->form_fields;

			// formatted the data.
			$this->form_fields = apply_filters( 'everest_forms_process_filter', $this->form_fields, $entry, $this->form_data );

			$this->form_fields = apply_filters( 'everest_forms_process_after_filter', $this->form_fields, $entry, $this->form_data );

			// Success - add entry to database.
			$entry_id = $this->entry_save( $this->form_fields, $entry, $this->form_data['id'], $this->form_data );

			if ( empty( $entry_id ) ) {
				// Error response.
				wp_send_json_error(
					array(
						'message' => __( 'Sorry! something went weong.', 'everest-forms-save-and-continue' ),
						'title'   => __( 'We could not save your entry, please try again.', 'everest-forms-save-and-continue' ),
					)
				);
			}

			$this->form_data['url'] = $entry['url'] . ( preg_match( '/\?/', $entry['url'] ) ? '&' : '?' ) . 'save-and-continue-token=' . $entry['token'];

			// Success response.
			wp_send_json_success(
				array(
					'message'  => apply_filters(
						'everest_forms_process_smart_tags',
						$this->form_data['settings']['save_and_continue_confirmation_message'],
						$this->form_data,
						$this->form_fields,
						$entry_id
					),
					'title'    => $this->form_data['settings']['save_and_continue_confirmation_title'],
					'url'      => $this->form_data['url'],
					'entry_id' => $entry_id,
				)
			);

		} catch ( Exception $e ) {

			// Error response.
			wp_send_json_error(
				array(
					'message' => __( 'Sorry! something went weong.', 'everest-forms-save-and-continue' ),
					'title'   => __( 'We could not save your entry, please try again.', 'everest-forms-save-and-continue' ),
				)
			);
		}
	}
}
