<?php
/**
 * Save Form Dataclass.
 *
 * @package EverestForms\SaveAndContinue
 * @since   1.0.0
 */

namespace EverestForms\SaveAndContinue;

/**
 * Save and Continue Feature Class.
 *
 * @since 1.0.0
 */
class DisplayFields extends \EVF_Form_Fields {

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_filter( 'everest_forms_field_properties', array( $this, 'process_save_and_continue_value' ), 10, 3 );
		add_filter( 'everest_forms_sc_add_form_value', array( $this, 'add_field_value' ), 10, 3 );
	}

	/**
	 * Process_save_and_continue_value.
	 *
	 * @param mixed $properties Field Properties.
	 * @param mixed $field Field.
	 * @param mixed $entry_field Entry Field.
	 *
	 * @since 1.0.0
	 */
	public function add_field_value( $properties, $field, $entry_field ) {

		switch ( $field['type'] ) {
			case 'select':
				$value_choices = ! empty( $entry_field['value_raw'] ) ? $entry_field['value_raw'] : array();
				$this->remove_field_choices_defaults( $field, $properties );
				if ( is_array( $value_choices ) ) {
					foreach ( $value_choices as $input => $single_value ) {
						$properties = $this->get_single_field_property_value( $single_value, sanitize_key( $input ), $properties, $field );
					}
				}
				break;

			case 'radio':
				$value = isset( $entry_field['value_raw'] ) ? $entry_field['value_raw'] : '';
				$this->remove_field_choices_defaults( $field, $properties );

				if ( '' !== $value ) {
					$properties = $this->get_single_field_property_value( $value, 'primary', $properties, $field );
				}
				break;
			case 'address':
				$input_keys = array( 'address1', 'address2', 'city', 'state', 'postal', 'country' );

				foreach ( $input_keys as $input_key ) {
					if ( isset( $entry_field[ $input_key ] ) ) {
						$properties = $this->get_single_field_property_value( $entry_field[ $input_key ], $input_key, $properties, $field );
					}
				}
				break;

			case 'checkbox':
				$value_choices = ! empty( $entry_field['value_raw'] ) ? $entry_field['value_raw'] : array();
				$this->remove_field_choices_defaults( $field, $properties );
				foreach ( $value_choices as $input => $single_value ) {
					$properties = $this->get_single_field_property_value( $single_value, sanitize_key( $input ), $properties, $field );
				}
				break;

			default:
				$value = isset( $entry_field['value'] ) ? $entry_field['value'] : '';
				if ( '' !== $value ) {
					$properties = $this->get_single_field_property_value( $value, 'primary', $field['properties'], $field );
				}
				break;
		}

		return $properties;
	}

	/**
	 * Process Value.
	 *
	 * @param string $properties Value.
	 * @param string $field  Field.
	 * @param mixed  $form_data Form Data.
	 *
	 * @since 1.0.0
	 */
	public function process_save_and_continue_value( $properties, $field, $form_data ) {
		global $wpdb;

		if ( ! isset( $_GET['save-and-continue-token'] ) || empty( $_GET['save-and-continue-token'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			return $properties;
		}

		if ( ! array_key_exists( 'enable_save_and_continue', $form_data['settings'] ) || '1' !== $form_data['settings']['enable_save_and_continue'] ) {
			return $properties;
		}

		$entry_token = sanitize_title( wp_unslash( $_GET['save-and-continue-token'] ) ); // phpcs:ignore WordPress.Security.NonceVerification

		$entry = wp_cache_get( $entry_token, 'evf-entry-save-and-continue' );
		if ( false === $entry ) {
			$entry = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}evf_entries WHERE token = %s LIMIT 1;", $entry_token ) );
			wp_cache_add( $entry_token, $entry, 'evf-entry-save-and-continue' );
		}

		if ( empty( $entry ) ) {
			return $properties;
		}

		$form_id  = $entry->form_id;
		$entry_id = $entry->entry_id;

		$form_data = evf()->form->get( $form_id, array( 'content_only' => true ) );

		$hide_empty   = isset( $_COOKIE['everest_forms_entry_hide_empty'] ) && 'true' === $_COOKIE['everest_forms_entry_hide_empty'];
		$entry_fields = apply_filters( 'everest_forms_entry_single_data', evf_decode( $entry->fields ), $entry, $form_data );

		// Check if the field can be displayed.
		if ( ! empty( $field['type'] ) && in_array( $field['type'], (array) apply_filters( 'everest_forms_entries_edit_fields_hidden', array( 'title', 'html' ) ), true ) ) {
			return $properties;
		}

		$field_id = $field['id'];

		$entry_field = ! empty( $entry_fields[ $field_id ] ) ? $entry_fields[ $field_id ] : array(
			'name'      => ! empty( $properties['label'] ) ? $properties['label'] : '',
			'value'     => '',
			'value_raw' => '',
			'id'        => ! empty( $properties['id'] ) ? $properties['id'] : '',
			'type'      => ! empty( $properties['type'] ) ? $properties['type'] : '',
		);

		$field_value         = ! empty( $entry_field['value'] ) ? $entry_field['value'] : '';
		$field_value         = apply_filters( 'everest_forms_html_field_value', $field_value, $entry_field, $form_data, 'entry-single' );
		$field['properties'] = $properties;
		// Display entry editable and non-editable form field.
		if ( apply_filters( 'everest_forms_admin_entries_edit_field_output_editable', $this->is_field_entries_editable( $field['type'] ), $field ) ) {
			$properties = apply_filters( 'everest_forms_sc_add_form_value', $field['properties'], $field, $entry_field );
		}

		return $properties;
	}

	/**
	 * Check if the field entries are editable.
	 *
	 * @param string $field_type Field type.
	 *
	 * @since 1.0.0
	 */
	private function is_field_entries_editable( $field_type ) {
		return (bool) apply_filters(
			'everest_forms_entries_field_editable',
			in_array(
				$field_type,
				array(
					'first-name',
					'last-name',
					'text',
					'textarea',
					'select',
					'radio',
					'checkbox',
					'number',
					'email',
					'url',
					'date-time',
					'phone',
					'address',
					'country',
					'range-slider',
					'rating',
				),
				true
			),
			$field_type
		);
	}

}
